package middleware

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/collaborators"
)

func makeKey(ss ...string) string {
	return strings.Join(ss, "-")
}

func CachingMiddleware(cache *cache.Cache) Middleware {
	return func(next service.Collaborators) service.Collaborators {
		return &cachingMiddleware{
			cache: cache,
			next:  next,
		}
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Collaborators
}

func (m cachingMiddleware) Set(ctx context.Context, spaceId, subject, role string) (err error) {

	err = m.next.Set(ctx, spaceId, subject, role)
	if err == nil {
		m.cache.Remove(spaceId)
		m.cache.Remove(subject)
	}
	return err
}

func (m cachingMiddleware) Get(ctx context.Context, spaceId, subject string) (role string, err error) {

	key := makeKey(spaceId, subject)
	value, e := m.cache.Get(key)
	if e == nil {
		return value.(string), err
	}
	role, err = m.next.Get(ctx, spaceId, subject)
	if err == nil {
		m.cache.Set(key, role)
	}
	return role, err
}

func (m cachingMiddleware) Remove(ctx context.Context, spaceId, subject string) (err error) {

	err = m.next.Remove(ctx, spaceId, subject)
	if err == nil {
		m.cache.Remove(makeKey(spaceId, subject))
		m.cache.Remove(spaceId)
		m.cache.Remove(subject)
	}
	return err
}

func (m cachingMiddleware) ListCollaborators(ctx context.Context, spaceId string) (collaborators []*service.Collaborator, err error) {

	value, e := m.cache.Get(spaceId)
	if e == nil {
		return value.([]*service.Collaborator), err
	}
	collaborators, err = m.next.ListCollaborators(ctx, spaceId)
	if err == nil {
		m.cache.Set(spaceId, collaborators)
	}
	return collaborators, err
}

func (m cachingMiddleware) ListSpaces(ctx context.Context, subject string) (collaborators []*service.Collaborator, err error) {

	value, e := m.cache.Get(subject)
	if e == nil {
		return value.([]*service.Collaborator), err
	}
	collaborators, err = m.next.ListSpaces(ctx, subject)
	if err == nil {
		m.cache.Set(subject, collaborators)
	}
	return collaborators, err
}
