package middleware

import (
	"context"
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/cache"
	service "git.perx.ru/perxis/perxis-go/pkg/clients"
)

func makeKey(ss ...string) string {
	return strings.Join(ss, "-")
}

func CachingMiddleware(cache *cache.Cache) Middleware {
	return func(next service.Clients) service.Clients {
		return &cachingMiddleware{
			cache: cache,
			next:  next,
		}
	}
}

type cachingMiddleware struct {
	cache *cache.Cache
	next  service.Clients
}

func (m cachingMiddleware) Create(ctx context.Context, client *service.Client) (cl *service.Client, err error) {

	cl, err = m.next.Create(ctx, client)
	if err == nil {
		m.cache.Remove(cl.SpaceID)
	}
	return cl, err
}

func (m cachingMiddleware) Get(ctx context.Context, spaceId string, id string) (cl *service.Client, err error) {

	key := makeKey(spaceId, id)
	value, e := m.cache.Get(key)
	if e == nil {
		return value.(*service.Client), err
	}
	cl, err = m.next.Get(ctx, spaceId, id)
	if err == nil {
		m.cache.Set(key, cl)
		for _, key := range keysFromIdentities(spaceId, cl) {
			m.cache.Set(key, cl)
		}
	}
	return cl, err
}

func (m cachingMiddleware) GetBy(ctx context.Context, spaceId string, params *service.GetByParams) (cl *service.Client, err error) {
	if params == nil {
		return m.next.GetBy(ctx, spaceId, params)
	}

	key := getIdentKey(spaceId, params)
	value, e := m.cache.Get(key)
	if e == nil {
		return value.(*service.Client), err
	}
	cl, err = m.next.GetBy(ctx, spaceId, params)
	if err == nil {
		m.cache.Set(makeKey(spaceId, cl.ID), cl)
		for _, key := range keysFromIdentities(spaceId, cl) {
			m.cache.Set(key, cl)
		}
	}
	return cl, err
}

func (m cachingMiddleware) List(ctx context.Context, spaceId string) (clients []*service.Client, err error) {

	value, e := m.cache.Get(spaceId)
	if e == nil {
		return value.([]*service.Client), err
	}
	clients, err = m.next.List(ctx, spaceId)
	if err == nil {
		m.cache.Set(spaceId, clients)
	}
	return clients, err
}

func (m cachingMiddleware) Update(ctx context.Context, client *service.Client) (err error) {

	err = m.next.Update(ctx, client)

	if err == nil {
		m.cache.Remove(client.SpaceID)
		value, e := m.cache.Get(makeKey(client.SpaceID, client.ID))
		if e == nil {
			client := value.(*service.Client)
			m.cache.Remove(makeKey(client.SpaceID, client.ID))
			for _, key := range keysFromIdentities(client.SpaceID, client) {
				m.cache.Remove(key)
			}
		}
	}
	return err
}

func (m cachingMiddleware) Delete(ctx context.Context, spaceId string, id string) (err error) {

	err = m.next.Delete(ctx, spaceId, id)
	if err == nil {
		value, e := m.cache.Get(makeKey(spaceId, id))
		if e == nil {
			client := value.(*service.Client)
			m.cache.Remove(makeKey(client.SpaceID, client.ID))
			for _, key := range keysFromIdentities(client.SpaceID, client) {
				m.cache.Remove(key)
			}
		}
		m.cache.Remove(spaceId)
	}
	return err
}

func (m cachingMiddleware) Enable(ctx context.Context, spaceId string, id string, enable bool) (err error) {

	err = m.next.Enable(ctx, spaceId, id, enable)
	if err == nil {
		value, e := m.cache.Get(makeKey(spaceId, id))
		if e == nil {
			client := value.(*service.Client)
			m.cache.Remove(makeKey(client.SpaceID, client.ID))
			for _, key := range keysFromIdentities(client.SpaceID, client) {
				m.cache.Remove(key)
			}
		}
		m.cache.Remove(spaceId)
	}
	return err
}

func keysFromIdentities(spaceID string, client *service.Client) []string {
	res := make([]string, 0)
	if client.APIKey != nil && client.APIKey.Key != "" {
		res = append(res, makeKey(spaceID, "api-key", client.APIKey.Key))
	}
	if client.TLS != nil && client.TLS.Subject != "" {
		res = append(res, makeKey(spaceID, "tls", client.TLS.Subject))
	}
	if client.OAuth != nil && client.OAuth.ClientID != "" {
		res = append(res, makeKey(spaceID, "oauth", client.OAuth.ClientID))
	}
	return res
}

func getIdentKey(spaceID string, params *service.GetByParams) string {
	switch {
	case params.APIKey != "":
		return makeKey(spaceID, "api-key", params.APIKey)
	case params.TLSSubject != "":
		return makeKey(spaceID, "tls", params.TLSSubject)
	case params.OAuthClientID != "":
		return makeKey(spaceID, "oauth", params.OAuthClientID)
	default:
		return ""
	}
}
