package auth

import (
	"context"

	"git.perx.ru/perxis/perxis-go/pkg/environments"
	"git.perx.ru/perxis/perxis-go/pkg/members"
	"git.perx.ru/perxis/perxis-go/pkg/permission"
	"git.perx.ru/perxis/perxis-go/pkg/roles"
)

type Principal interface {
	GetID(ctx context.Context) string
	IsValid(ctx context.Context) bool
	IsSystem(ctx context.Context) bool
	HasAccess(ctx context.Context, spID, orgID string) error

	IsManagementAllowed(ctx context.Context, spaceID string) error
}

type SpaceAccessor interface {
	Principal
	Space(spaceID string) SpaceAccessor

	// HasSpaceAccess проверяет, есть ли у принципала доступ на чтение пространства
	// (просмотр информации о пространстве, окружений, т.д. - доступ к записям коллекций
	// определяется отдельным набором правил, см. SpaceAccessor.Rules())
	HasSpaceAccess(ctx context.Context, spaceID string) bool
	HasEnvironmentAccess(ctx context.Context, spaceID, env string) bool

	// Member возвращает роль принципала в организации
	Member(ctx context.Context) members.Role

	Role(ctx context.Context, spaceID string) *roles.Role

	// Rules возвращает набор правил, по которым принципал может получить
	// доступ к записям коллекций пространства.
	Rules(ctx context.Context, spaceID, envID string) permission.Ruleset
}

type OrganizationAccessor interface {
	Principal
	Organization(orgID string) OrganizationAccessor
	Member(ctx context.Context) members.Role
}

func hasEnvironmentAccess(ctx context.Context, envsrv environments.Environments, role *roles.Role, envID string) bool {
	return role != nil && role.CanAccessEnvironment(ctx, &environments.Environment{SpaceID: role.SpaceID, ID: envID}, envsrv)
}
