package auth

import (
	"strings"

	"git.perx.ru/perxis/perxis-go/pkg/clients"
	"git.perx.ru/perxis/perxis-go/pkg/collaborators"
	"git.perx.ru/perxis/perxis-go/pkg/environments"
	"git.perx.ru/perxis/perxis-go/pkg/members"
	"git.perx.ru/perxis/perxis-go/pkg/roles"
	"git.perx.ru/perxis/perxis-go/pkg/spaces"
	"git.perx.ru/perxis/perxis-go/pkg/users"
)

type PrincipalFactory struct {
	users.Users
	members.Members
	collaborators.Collaborators
	roles.Roles
	clients.Clients
	spaces.Spaces
	environments.Environments
}

func (f PrincipalFactory) User(identity string) Principal {
	return &UserPrincipal{
		identity:      identity,
		users:         f.Users,
		members:       f.Members,
		roles:         f.Roles,
		collaborators: f.Collaborators,
		spaces:        f.Spaces,
		environments:  f.Environments,
	}
}

func (f PrincipalFactory) Client(param *clients.GetByParams) Principal {
	return &ClientPrincipal{
		identity: param,
		//authID:       authID,
		clients:       f.Clients,
		environments:  f.Environments,
		roles:         f.Roles,
		spaces:        f.Spaces,
		collaborators: f.Collaborators,
	}
}

func (f PrincipalFactory) Anonymous() Principal {
	return &Anonymous{
		roles:  f.Roles,
		spaces: f.Spaces,
	}
}

func (f PrincipalFactory) System() Principal {
	return &SystemPrincipal{}
}

func (f PrincipalFactory) Principal(principalId string) Principal {
	switch {
	case strings.Contains(principalId, "Subject="):
		return f.Client(&clients.GetByParams{TLSSubject: getSubject(principalId)})
	case strings.HasSuffix(principalId, "@clients"):
		return f.Client(&clients.GetByParams{OAuthClientID: strings.TrimSuffix(principalId, "@clients")})
	case strings.HasPrefix(principalId, "API-Key"):
		return f.Client(&clients.GetByParams{APIKey: strings.TrimPrefix(principalId, "API-Key ")})
	default:
		return f.User(principalId)
	}
}

func getSubject(header string) string {
	var p string
	for _, part := range strings.Split(header, ";") {
		if strings.Contains(part, "Subject") {
			p = strings.TrimSuffix(strings.TrimPrefix(part, "Subject=\""), "\"")
			break
		}
	}
	return p
}
