package auth

import (
	"context"
	"fmt"

	"git.perx.ru/perxis/perxis-go/pkg/environments"
	"git.perx.ru/perxis/perxis-go/pkg/errors"
	"git.perx.ru/perxis/perxis-go/pkg/members"
	"git.perx.ru/perxis/perxis-go/pkg/permission"
	"git.perx.ru/perxis/perxis-go/pkg/roles"
	"git.perx.ru/perxis/perxis-go/pkg/service"
	"git.perx.ru/perxis/perxis-go/pkg/spaces"
)

type Anonymous struct {
	roles        roles.Roles
	spaces       spaces.Spaces
	spaceID      string
	environments environments.Environments
}

func (Anonymous) GetID(ctx context.Context) string  { return "anonymous" }
func (Anonymous) IsValid(ctx context.Context) bool  { return false }
func (Anonymous) IsSystem(ctx context.Context) bool { return false }
func (Anonymous) IsManagementAllowed(ctx context.Context, spaceID string) error {
	return service.ErrAccessDenied
}

func (a Anonymous) Space(spaceID string) SpaceAccessor {
	a.spaceID = spaceID
	return &a
}

func (a *Anonymous) getSpace(ctx context.Context, spaceID string) *spaces.Space {
	if spaceID == "" {
		return nil
	}
	space, _ := a.spaces.Get(WithSystem(ctx), spaceID)
	return space
}

func (a *Anonymous) HasSpaceAccess(ctx context.Context, spaceID string) bool {
	if a.spaceID == "" || a.spaces == nil {
		return false
	}
	return a.Role(ctx, spaceID) != nil
}

func (a *Anonymous) Member(ctx context.Context) members.Role {
	return members.NotMember
}

func (a *Anonymous) Role(ctx context.Context, spaceID string) *roles.Role {
	if a.spaceID == "" || a.roles == nil {
		return nil
	}
	role, err := a.roles.Get(WithSystem(ctx), spaceID, roles.AnonymousRole)
	if err != nil {
		return nil
	}
	return role
}

func (a *Anonymous) Rules(ctx context.Context, spaceID, envID string) permission.Ruleset {
	role := a.Role(WithSystem(ctx), spaceID)
	if role == nil {
		return nil
	}

	if !a.HasEnvironmentAccess(ctx, spaceID, envID) {
		return nil
	}

	return role.Rules
}

func (a *Anonymous) HasEnvironmentAccess(ctx context.Context, space, env string) bool {
	return hasEnvironmentAccess(ctx, a.environments, a.Role(ctx, space), env)
}

func (Anonymous) Format(f fmt.State, verb rune) {
	f.Write([]byte("AnonymousPrincipal{}"))
}

func (a Anonymous) HasAccess(ctx context.Context, spaceID, orgID string) error {
	if !a.IsValid(ctx) {
		return service.ErrAccessDenied
	}

	if a.IsSystem(ctx) {
		return nil
	}

	if spaceID != "" {
		hasAllow, err := a.hasRole(ctx, spaceID)
		if err != nil {
			return err
		}

		if hasAllow {
			return nil
		}
	}

	if a.Member(ctx).IsPrivileged() {
		return nil
	}

	return service.ErrAccessDenied
}

func (a *Anonymous) hasRole(ctx context.Context, spaceID string) (bool, error) {
	if a.spaceID == "" || a.roles == nil {
		return false, nil
	}
	_, err := a.roles.Get(WithSystem(ctx), spaceID, roles.AnonymousRole)
	if err == nil {
		return true, nil
	}

	if errors.Is(err, service.ErrNotFound) {
		if sp := a.getSpace(ctx, spaceID); sp == nil {
			return false, service.ErrNotFound
		}
	}
	return false, nil
}
